/**
 * TwoPointHalfCrossover.java
 * 
 * Class representing a two point crossover operator on half of the variables
 * only. The other half is treated as voltage levels and copied together with
 * corresponding assignments of the first part.
 * Based on TwoPointsCrossover from jMetal.
 * 
 * @author Mateusz Guzek
 * @version 1.0
 */
package greenmetal.base.operator.crossover;

import java.util.Properties;
import jmetal.base.*;
import jmetal.base.operator.crossover.Crossover;
import jmetal.base.variable.*;
import jmetal.base.*;
import jmetal.util.Configuration;
import jmetal.util.JMException;
import jmetal.util.PseudoRandom;
import jmetal.util.Configuration.*;

/**
 * This class allows to apply a Single Point crossover operator using two parent
 * solutions.
 * NOTE: the operator is applied to integer solutions, considering first
 * half of the solution as a crossover eligible solution part.
 * @author Mateusz Guzek
 */
public class TwoPointHalfCrossover extends Crossover {

	/**
	 * INT_SOLUTION represents class jmetal.base.solutionType.IntSolutionType
	 */
	private static Class INT_SOLUTION ; 

	/**
	 * Constructor
	 * Creates a new instance of the single point crossover operator
	 */
	public TwoPointHalfCrossover() {
		System.out.println("Two point EMcrossover initialized");
		try {
			INT_SOLUTION = Class.forName("jmetal.base.solutionType.IntSolutionType") ;
		} catch (ClassNotFoundException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} // catch
	} // SinglePointCrossover


	/**
	 * Constructor
	 * Creates a new instance of the single point crossover operator
	 */
	public TwoPointHalfCrossover(Properties properties) {
		this();
	} // SinglePointCrossover

	/**
	 * Perform the crossover operation.
	 * @param probability Crossover probability
	 * @param parent1 The first parent
	 * @param parent2 The second parent
	 * @return An array containig the two offsprings
	 * @throws JMException
	 */
	public Solution[] doCrossover(double probability,
			Solution parent1,
			Solution parent2) throws JMException {
		Solution[] offSpring = new Solution[2];
		offSpring[0] = new Solution(parent1);
		offSpring[1] = new Solution(parent2);
		try {
			if (PseudoRandom.randDouble() < probability) {
				// Integer representation
				int valueX1;
				int valueX2;
				int valueY1;
				int valueY2;
				//if(PseudoRandom.randDouble() < 0.5){
					//Crossover first variable
					int crossoverPoint1 = PseudoRandom.randInt(0, parent1.numberOfVariables()/2);
					int crossoverPoint2 = PseudoRandom.randInt(0, parent1.numberOfVariables()/2);
					if(crossoverPoint1>crossoverPoint2){
						int tmp = crossoverPoint1;
						crossoverPoint1 = crossoverPoint2;
						crossoverPoint2 = tmp;
					}
//					System.out.println("CP1: "+crossoverPoint1+" CP2 "+crossoverPoint2);
					for (int i = crossoverPoint1; i < crossoverPoint2/*parent1.numberOfVariables()/2*/; i++) {
						valueX1 = (int) parent1.getDecisionVariables()[i].getValue();
						valueX2 = (int) parent2.getDecisionVariables()[i].getValue();
						valueY1 = (int) parent1.getDecisionVariables()[i + parent1.numberOfVariables()/2].getValue();
						valueY2 = (int) parent2.getDecisionVariables()[i + parent1.numberOfVariables()/2].getValue();
						offSpring[0].getDecisionVariables()[i].setValue(valueX2);
						offSpring[1].getDecisionVariables()[i].setValue(valueX1);
						offSpring[0].getDecisionVariables()[i+parent1.numberOfVariables()/2].setValue(valueY2);
						offSpring[1].getDecisionVariables()[i+parent1.numberOfVariables()/2].setValue(valueY1);
					} // for
//				} else {
//					int crossoverPoint = PseudoRandom.randInt(parent1.numberOfVariables()/2, parent1.numberOfVariables() - 1);
//					for (int i = crossoverPoint; i < parent1.numberOfVariables(); i++) {
//						valueX1 = (int) parent1.getDecisionVariables()[i].getValue();
//						valueX2 = (int) parent2.getDecisionVariables()[i].getValue();
//						offSpring[0].getDecisionVariables()[i].setValue(valueX2);
//						offSpring[1].getDecisionVariables()[i].setValue(valueX1);
//					}
					// Int representation
				
			}
		} catch (ClassCastException e1) {
			Configuration.logger_.severe("SinglePointCrossover.doCrossover: Cannot perfom " +
			"SinglePointCrossover");
			Class cls = java.lang.String.class;
			String name = cls.getName();
			throw new JMException("Exception in " + name + ".doCrossover()");
		}
		return offSpring;
	} // doCrossover

	/**
	 * Executes the operation
	 * @param object An object containing an array of two solutions
	 * @return An object containing an array with the offSprings
	 * @throws JMException
	 */
	public Object execute(Object object) throws JMException {
		Solution[] parents = (Solution[]) object;

		if (((parents[0].getType().getClass() != INT_SOLUTION) ||
								(parents[1].getType().getClass() != INT_SOLUTION))) {

			Configuration.logger_.severe("SinglePointCrossover.execute: the solutions " +
					"are not of the right type. The type should be 'Binary' or 'Int', but " +
					parents[0].getType() + " and " +
					parents[1].getType() + " are obtained");

			Class cls = java.lang.String.class;
			String name = cls.getName();
			throw new JMException("Exception in " + name + ".execute()");
		} // if

		Double probability = (Double) getParameter("probability");
		if (parents.length < 2) {
			Configuration.logger_.severe("SinglePointCrossover.execute: operator " +
			"needs two parents");
			Class cls = java.lang.String.class;
			String name = cls.getName();
			throw new JMException("Exception in " + name + ".execute()");
		} else if (probability == null) {
			Configuration.logger_.severe("SinglePointCrossover.execute: probability " +
			"not specified");
			Class cls = java.lang.String.class;
			String name = cls.getName();
			throw new JMException("Exception in " + name + ".execute()");
		}

		Solution[] offSpring;
		offSpring = doCrossover(probability.doubleValue(),
				parents[0],
				parents[1]);

		//-> Update the offSpring solutions
		for (int i = 0; i < offSpring.length; i++) {
			offSpring[i].setCrowdingDistance(0.0);
			offSpring[i].setRank(0);
		}
		return offSpring;//*/
	} // execute
} // SinglePointCrossover
