/**
 * NSGAII_main_EMscheduling.java
 *
 * EMscheduling class for NSGAII.
 * Based on NSGAII_main from jMetal.
 *
 * @author Juan J. Durillo
 * @author Antonio J. Nebro
 * @author Mateusz Guzek
 * @version 1.0
 *   This implementation of NSGA-II makes use of a QualityIndicator object
 *   to obtained the convergence speed of the algorithm. This version is used
 *   in the paper:
 *     A.J. Nebro, J.J. Durillo, C.A. Coello Coello, F. Luna, E. Alba 
 *     "A Study of Convergence Speed in Multi-Objective Metaheuristics." 
 *     To be presented in: PPSN'08. Dortmund. September 2008.
 *     
 *   Besides the classic NSGA-II, a steady-state version (ssNSGAII) is also
 *   included (See: J.J. Durillo, A.J. Nebro, F. Luna and E. Alba 
 *                  "On the Effect of the Steady-State Selection Scheme in 
 *                  Multi-Objective Genetic Algorithms"
 *                  5th International Conference, EMO 2009, pp: 183-197. 
 *                  April 2009)
 *   
 */
package greenmetal.metaheuristics.nsgaII;

import greenmetal.base.operator.crossover.*;
import greenmetal.base.operator.mutation.EMMutationFactory;
import greenmetal.problems.EMscheduling.EMscheduling;
import jmetal.base.*;
import jmetal.base.operator.mutation.*    ; 
import jmetal.base.operator.selection.*   ;
import jmetal.metaheuristics.nsgaII.NSGAII;
import jmetal.problems.*                  ;
import jmetal.problems.DTLZ.*;
import jmetal.problems.ZDT.*;
import jmetal.problems.WFG.*;
import jmetal.problems.LZ09.* ;

import jmetal.util.Configuration;
import jmetal.util.JMException;

import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.io.StreamTokenizer;

import java.util.logging.FileHandler;
import java.util.logging.Logger;

import jmetal.qualityIndicator.QualityIndicator;

public class NSGAII_main_EMscheduling {
  public static boolean test_evaluation = false;
	
  public static Logger      logger_ ;      // Logger object
  public static FileHandler fileHandler_ ; // FileHandler object

  /**
   * @param args Command line arguments.
   * @throws JMException 
   * @throws IOException 
   * @throws SecurityException 
   * Usage: three options
   *      - jmetal.metaheuristics.nsgaII.NSGAII_main
   *      - jmetal.metaheuristics.nsgaII.NSGAII_main problemName
   *      - jmetal.metaheuristics.nsgaII.NSGAII_main problemName paretoFrontFile
   */
  public static void main(String [] args) throws 
                                  JMException, 
                                  SecurityException, 
                                  IOException, 
                                  ClassNotFoundException {
    Problem   problem   ;         // The problem to solve
    Algorithm algorithm ;         // The algorithm to use
    Operator  crossover ;         // Crossover operator
    Operator  mutation  ;         // Mutation operator
    Operator  selection ;         // Selection operator
    
    QualityIndicator indicators ; // Object to get quality indicators

    // Logger object and file to store log messages
    logger_      = Configuration.logger_ ;
    fileHandler_ = new FileHandler("NSGAII_main.log"); 
    logger_.addHandler(fileHandler_) ;
    
    indicators = null ;
 
    problem =  new EMscheduling().initializeEMProblem(args,true, logger_, test_evaluation);
    if(problem==null){
    	logger_.severe("ERROR: No problem!");
    }
    
    algorithm = new NSGAII(problem);
    if(test_evaluation){
    	EMscheduling ems = (EMscheduling) problem;
    	Solution test_solution = new Solution(problem);


    	FileReader fr = new FileReader(new File(args[2]));
    	BufferedReader br = new BufferedReader(fr);
    	String sol = br.readLine();
    	String[] nums = sol.split("\\s");

    	Variable[] var =  test_solution.getDecisionVariables();
    	System.out.print("Decision variables:\nPROC ");
    	for(int i=0; i < var.length;i++){
    		var[i].setValue(Double.parseDouble(nums[i]));
    		System.out.print(" "+var[i]);
    		if(i==var.length/2-1)
    			System.out.print("\nVOLT ");
    	}

    	test_solution.setDecisionVariables(var);
    	System.out.println("");
    	double[] test_result = ems.getLocalScheduler().evaluateSolution(test_solution);
    	System.out.println("\nObjective functions:");
    	for(int i=0; i < test_result.length;i++){
    		System.out.println("O"+i+": "+test_result[i]);    	
    	}
    } else {
    //algorithm = new ssNSGAII(problem);
    
    // Algorithm parameters
    algorithm.setInputParameter("populationSize",100);
    algorithm.setInputParameter("maxEvaluations",5000);
    
    // Mutation and Crossover for Integer codification 
    crossover = EMCrossoverFactory.getCrossoverOperator("GroupSinglePointHalfCrossover"); 
//    crossover = EMCrossoverFactory.getCrossoverOperator("SinglePointHalfCrossover");
    crossover.setParameter("probability",0.8);                   
    crossover.setParameter("distributionIndex",20.0);//20
  
    mutation = EMMutationFactory.getMutationOperator("EMBitFlipMutation");                    
    mutation.setParameter("probability",1.0/problem.getNumberOfVariables());
 	mutation.setParameter("distributionIndex",20.0);//20  

    // Selection Operator 
    selection = SelectionFactory.getSelectionOperator("BinaryTournament2") ;                           

    // Add the operators to the algorithm
    algorithm.addOperator("crossover",crossover);
    algorithm.addOperator("mutation",mutation);
    algorithm.addOperator("selection",selection);

    // Add the indicator object to the algorithm
    algorithm.setInputParameter("indicators", indicators) ;
    
    // Execute the Algorithm
    long initTime = System.currentTimeMillis();
    SolutionSet population = algorithm.execute();
    long estimatedTime = System.currentTimeMillis() - initTime;
    
    // Result messages 
    logger_.info("Total execution time: "+estimatedTime + "ms");
    logger_.info("Variables values have been writen to file VAR");
    population.printVariablesToFile("VAR");    
    logger_.info("Objectives values have been writen to file FUN");
    population.printObjectivesToFile("FUN");
  
    if (indicators != null) {
      logger_.info("Quality indicators") ;
      logger_.info("Hypervolume: " + indicators.getHypervolume(population)) ;
      logger_.info("GD         : " + indicators.getGD(population)) ;
      logger_.info("IGD        : " + indicators.getIGD(population)) ;
      logger_.info("Spread     : " + indicators.getSpread(population)) ;
      logger_.info("Epsilon    : " + indicators.getEpsilon(population)) ;  
     
      int evaluations = ((Integer)algorithm.getOutputParameter("evaluations")).intValue();
      logger_.info("Speed      : " + evaluations + " evaluations") ;      
    } // if
    }
  } //main
} // NSGAII_main
