package greenmetal.problems.EMscheduling;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.logging.Logger;
import java.lang.Math;

import jmetal.base.Solution;
import jmetal.util.JMException;
/**
 * EstScheduler.java
 * 
 * Scheduler used during evaluation. It returns information about a given schedule:
 *  its length and energy used during execution.
 *  
 * @author Mateusz Guzek
 * @version 1.0
 */
public class EstScheduler {
	public static double small_value = 0.00001;
	public static int count = 0;
	public static PairVLSSpeed[][] processorsVoltagesGlobal;
	protected  boolean test_evaluation;
	private Graphe DAG;
	/**
	 * Task table ordered by decreased b_level value, used to schedule tasks.
	 */
	private Task[] sortedTasks;
	/**
	 * Table of Scheduled task objects, which represent assigned Tasks of DAG
	 */
	private ScheduledTask[] tasksToSchedule;
	/**
	 * Table of lists of scheduled tasks, representing schedules of processors.
	 */
	private List<ScheduledTask> processorsLists[];
	/**
	 * Possible VLS setting for processors
	 */
	private PairVLSSpeed[][] processorPairs;
	double used_times[];
	private Logger		logger_;
	private boolean debug_once;
	private boolean reclamation;


	public EstScheduler(Graphe DAG, PairVLSSpeed[][] processorPairs,boolean reclamation, Logger logger_,boolean test_evaluation){
		super();
		this.reclamation = reclamation;
		this.test_evaluation = test_evaluation;
		this.DAG = DAG;
		this.processorPairs = processorPairs;
		processorsVoltagesGlobal = processorPairs;
		this.logger_ = logger_;
		sortedTasks = DAG.getTaskTableCopy();
		tasksToSchedule = new ScheduledTask[DAG.getTaskNumber()];
		processorsLists =(ArrayList<ScheduledTask>[]) new ArrayList[DAG.getMaxClusters()];
		for(int i = 0; i < processorsLists.length; i++){
			processorsLists[i]= new ArrayList<ScheduledTask>();
		}
		used_times = new double[processorsLists.length];

		/*Calculation of b_level and then sorting tasks according to it
		 *  with decreasing order.*/
		this.main_Blevel();
		this.sortTasks(new B_levelComparator());
	}
	double bottomLevel(Task x, double blevel)
	{
		double max = 0.0;
		double comm= 0.0;
		double max_blevel =0.0;
		boolean band = false;
		int i;
		Edge pastEdge;	
		if (x.getNumOutEdges()>0) {
			for (i=0; i<x.getNumOutEdges(); i++) {
				pastEdge = x.getOutEdge(i);
				if (pastEdge.getEndTask().getB_level()!= 0.0) {
					comm = pastEdge.getCommTime()+ pastEdge.getEndTask().getB_level();

					if(comm > max)
						max = comm;
					band = true;
				}
				else {
					band = false;
					return 0.0;
				}
			}
		}else {
			return x.getAvgRunTime();
		}
		if (band) {
			max_blevel = x.getAvgRunTime() + max;
			return max_blevel;
		}
		else{
			return 0.0;
		}
	}

	public double main_Blevel()
	{
		double b_level = 0.0;
		double maxBlevel=0.0;
		double[] blevelT = new double[DAG.getTaskNumber()];
		int i,j, nb_tasks=0;

		for(i=0; i<DAG.getTaskNumber(); i++){
			blevelT[i]=0.0;
		}
		while (nb_tasks<DAG.getTaskNumber()) {

			for(i=DAG.getTaskNumber()-1, j=0; i>=0; --i, j++)
			{
				if (blevelT[j]==0.0) {
					b_level = bottomLevel(DAG.getTask(i) , b_level);
					if(b_level != 0.0){
						if (b_level > maxBlevel)
							maxBlevel = b_level;

						blevelT[j]=b_level;
						DAG.getTask(i).setB_level(b_level);
						b_level = 0.0;
						nb_tasks++;
					}
				}
			}
		}

		return maxBlevel;
	}

	public double[] evaluateSolution(Solution solution){
		double[] result; 
		result=getScheduleLength(solution);
		return result;
	}

	public void sortTasks(Comparator<Task> comparator){
		Arrays.sort(sortedTasks, comparator);
		int taskNumber = sortedTasks.length;
		Task[] tmpSortedTasks = new Task[taskNumber];
		for(int i = 0; i < taskNumber; i++){
			tmpSortedTasks[i]=sortedTasks[taskNumber-i-1];
		}
		sortedTasks = tmpSortedTasks;

	}
	private double[] getScheduleLength(Solution solution){
		/*VLS pair used to compute idle time energy (does not have to be real 
		 * DVFS level, may be sleep mode characteristic encapsulated)*/
		//PairVLSSpeed idlePair = processorPairs[processorPairs.length-1];
		for(int i =0; i< used_times.length; i++){
			used_times[i]=0;
		}
		double[] result = new double[2];
		for(int i = 0; i < processorsLists.length; i++){
			processorsLists[i].clear();
		}
		for(int i = 0; i < DAG.getTaskNumber(); i++){
			try {
				tasksToSchedule[i] = new  ScheduledTask(
						(int)solution.getDecisionVariables()[sortedTasks[i].getIndex()].getValue(),
						sortedTasks[i],
						processorPairs[(int)solution.getDecisionVariables()[sortedTasks[i].getIndex()].getValue()]
						               [(int)solution.getDecisionVariables()[sortedTasks[i].getIndex()+DAG.getTaskNumber()].getValue()]);
			} catch (JMException e) {
				e.printStackTrace();
			}
		}
		for(int i = 0; i < DAG.getTaskNumber(); i++){
			scheduleTask(tasksToSchedule[i]);
		}
		if(test_evaluation){
			printScheduling();
		}
		/*MAKESPAN:*/
		double totalExecutionTime=0;
		int k=-1;
		for(int i = 0; i < processorsLists.length; i++){
			if(!processorsLists[i].isEmpty()){
				result[0]=processorsLists[i].get(processorsLists[i].size()-1).getEndTime();
				if(debug_once){System.out.println("Execution proc[0]:"+result[0]);}
				totalExecutionTime=result[0];
				k = i+1;
				break;
			}
		}
		if(k==-1){
			logger_.severe("ERROR: all processors are empty.");
		}
		for(int i = k; i < processorsLists.length; i++){
			if(!processorsLists[i].isEmpty()){
				double tmpResult = processorsLists[i].get(processorsLists[i].size()-1).getEndTime();
				if(debug_once){System.out.println("Execution proc["+i+"]:"+tmpResult);}

				//if(result[0] == -1 || tmpResult > result[0]){
				if( result[0] < tmpResult){
					result[0] = tmpResult;
				}
				totalExecutionTime += tmpResult;
			}
		}
		if(debug_once)printScheduling();//System.out.print(this.getStringScheduledTasks());
		/*ENERGY:*/
		calculateSlackInSchedule(result[0]);
		if(reclamation){
			result[1] = getProcessingEnergySlackReclamation(result[0],processorsLists.length,processorPairs);
//			double test = getProcessingEnergy(result[0],processorsLists.length,processorPairs);
//			if(result[1]-test>small_value){
//				logger_.severe("Energy with DVFS bigger than without! "+ ++count);
//				System.out.println("DVFS " + result[1]+ " No DVFS " + test);
//			}
		} else {
			result[1]= getProcessingEnergy(result[0],processorsLists.length,processorPairs);
		}
		/* Set all Task.tmpScheduledTask to null. Not neccesary, but done to 
		 * reassure that tasks in DAG are not influenced by old scheduling.*/
		DAG.resetAllTasks();
		debug_once = false;
		return result;

	}
	private void scheduleTask(ScheduledTask tmpScheduledTask){
		int position = findPlaceForScheduledTask(processorsLists[tmpScheduledTask.getUsedProcessor()], tmpScheduledTask);
		if(position == -1){
			logger_.info("Local scheduler error: no place found to insert task");
			return;
		}
		tmpScheduledTask.setPositionInProcessor(position);
		processorsLists[tmpScheduledTask.getUsedProcessor()].add(position, tmpScheduledTask);
		used_times[tmpScheduledTask.getUsedProcessor()]+=tmpScheduledTask.getProccesingTime();
		//System.out.println("Used time proc:"+ tmpScheduledTask.getUsedProcessor()+ "=" +used_times[tmpScheduledTask.getUsedProcessor()]);
	}
	private int findPlaceForScheduledTask(List<ScheduledTask> taskList, ScheduledTask scheduledTask){
		Task t = scheduledTask.getTask();
		/*Source task:*/
		if(t.getNumInEdges()==0){
			return findFirstFreePosition(taskList,0, scheduledTask);
		} 
		/*Non source task, do something more complicated:*/
		/* non source task. Find the earliest start time by taking the maximum of all
		predecessors finishing times plus the communication time. If the predecessor
		task is on the same processor, the communication time is zero. */
		else {
			double earliestStart = -1;
			for(int i = 0; i < t.getNumInEdges(); i++){
				double commTime;
				double startTime;
				Edge pastEdge = t.getInEdge(i);
				Task previousTask = pastEdge.getStartTask();
				ScheduledTask previousScheduledTask = previousTask.getTmpScheduledTask();
				if(scheduledTask.getUsedProcessor()==previousScheduledTask.getUsedProcessor()){
					commTime = 0;
				} else {
					commTime = pastEdge.getCommTime();
				}
				startTime = previousScheduledTask.getEndTime()+commTime;
				if(earliestStart == -1 || earliestStart < startTime){
					earliestStart = startTime;
				}

			}
			//BUG: possibility to execute two task at the same time, when the 
			//constraints are the communications.

			return findFirstFreePosition(taskList, earliestStart, scheduledTask);

		}

	}

	private int findFirstFreePosition(List<ScheduledTask> taskList, double earliestTime, ScheduledTask scheduledTask){
		double previousEndTime = 0;
		double timeInterval;
		int i;
		if(taskList == null){
			logger_.severe("No processor task list.");
			return -1;
		}
		if(earliestTime ==-1){
			logger_.severe("ERROR: EST == -1");
		}
		if(taskList.isEmpty()){
			scheduledTask.setStartTime(earliestTime);
			return 0;
		}
		for(i = 0; i < taskList.size(); i++){
			if(taskList.get(i).getStartTime()>earliestTime){
				timeInterval = java.lang.Math.min(taskList.get(i).getStartTime() - previousEndTime,
						taskList.get(i).getStartTime() - earliestTime);

				if(timeInterval>=scheduledTask.getProccesingTime()){
					scheduledTask.setStartTime(java.lang.Math.max(previousEndTime,earliestTime));
					return i;
				}
			}
			previousEndTime = taskList.get(i).getEndTime();

		}

		scheduledTask.setStartTime(java.lang.Math.max(previousEndTime,earliestTime));
		return i;
	}
	//System.out.println("earliestTime:"+earliestTime);
	//java.lang.Math.max(previousEndTime,scheduledTask.getStartTime());
	//System.out.println("Time interval:" + timeInterval);
	/* */
	String getStringPairsVLSSpeed(){
		String result = "";
		for(int i = 0; i < processorPairs.length; i++){
			result += "Pair" + i + "\n"; 
			for(int j = 0 ; j < processorPairs[i].length; j++){
				result += processorPairs[i][j].toString() + "\n";
			}
		}
		return result;
	}
	String getStringScheduledTasks(){
		String result = "";
		for(int i = 0 ; i < tasksToSchedule.length; i++){
			result += tasksToSchedule[i].toString()+"\n";
		}
		return result;
	}

	void printScheduling(){				
		for(int i = 0 ; i < tasksToSchedule.length; i++){
			System.out.format("T %d"+" P: %d"
					+ " Rsp: %.2f"
					+" start: %.2f"
					+" base: %.2f"
					+" exec: %.2f"
					+" slack: %.2f"
					+ " end: %.2f\n",
					tasksToSchedule[i].getTask().getIndex(),
					tasksToSchedule[i].getUsedProcessor(),
					tasksToSchedule[i].getVlsSettings().getrSpeed(),
					tasksToSchedule[i].getStartTime(),
					tasksToSchedule[i].getTask().getRunTime(tasksToSchedule[i].getUsedProcessor()),
					tasksToSchedule[i].getProccesingTime(),
					tasksToSchedule[i].getSlack(),
					tasksToSchedule[i].getEndTime());
		}
	}
	private void printProcessor(double makespan, int i) {

		for(int j = 0; j < processorsLists[i].size(); j++){
			System.out.format("T %d"+" P: %d"
					+ " Rsp: %.2f"
					+" start: %.2f"
					+" base: %.2f"
					+" exec: %.2f"  
					+ " end: %.2f\n",
					processorsLists[i].get(j).getTask().getIndex(),
					processorsLists[i].get(j).getUsedProcessor(),
					processorsLists[i].get(j).getVlsSettings().getrSpeed(),
					processorsLists[i].get(j).getStartTime(),
					processorsLists[i].get(j).getTask().getRunTime(processorsLists[i].get(j).getUsedProcessor()),
					processorsLists[i].get(j).getProccesingTime(),
					processorsLists[i].get(j).getEndTime());
		}
		System.exit(1);
	}
	double getProcessingEnergy(double makespan,int n,PairVLSSpeed[][] processorPairs){
		double result = 0;
		//double total_idle = 0;
		for(int i = 0; i < tasksToSchedule.length; i++){
			result += tasksToSchedule[i].getComputationEnergy();
		}
		for(int i = 0; i < processorsLists.length ; i++){
			if(makespan+0.00001<used_times[i]){
				logger_.severe("ERROR: Makespan smaller than used time, proc: "+i+ " mksp: "
						+ makespan+" used time["+i+"] "+ used_times[i]+".\n");
				printProcessor(makespan, i);
			}
			if(test_evaluation){System.out.println("Time used by proc"+i+" = "+used_times[i]);}
			//total_idle+=makespan-used_times[i];
			result += ScheduledTask.getEnergy(makespan-used_times[i], processorPairs[i][processorPairs[i].length-1]);
		}
	//	System.out.println("Idle old ver. "+total_idle);
		//		return total_idle;
		return result;
	}
	double getProcessingEnergySlackReclamation(double makespan,int n,PairVLSSpeed[][] processorPairs){
		double result = 0;
		double total_idle = 0;
		//calculateSlackInSchedule(makespan);
		//Do voltage scaling for each task and compute its energy
		for(int i = 0; i < processorsLists.length ; i++){

			double tmp_idle = makespan;
			//double test_idle = makespan;
			double r_min = processorPairs[i][processorPairs[i].length-1].getrSpeed();
			for(int j = 0; j < processorsLists[i].size(); j++){
				if(processorsLists[i].get(j).getSlack() == -1){
					logger_.severe("EstScheduler.getProcessingEnergySlackReclamation: Slack computed in a bad way!");
				}
				double t_min; // time in which min voltage is used
				double t_c;	// time in which chosen voltage is used
				double t_b = processorsLists[i].get(j).getBaseExecutionTime();
				if(processorsLists[i].get(j).getVlsSettings().getrSpeed() - r_min>0){
					t_c = (t_b - r_min * processorsLists[i].get(j).getSlack())
					/ (processorsLists[i].get(j).getVlsSettings().getrSpeed() - r_min);
				} else {
					t_c = 0;
				}
				if(t_c<=0){
					t_c = 0;
					t_min = t_b/r_min;
				} else {
					t_min = processorsLists[i].get(j).getSlack() - t_c;
				}
				if(t_c>0){
					result += ScheduledTask.getEnergy(t_c, (processorsLists[i].get(j).getVlsSettings()));
				}
				result += ScheduledTask.getEnergy(t_min, processorPairs[i][processorPairs[i].length-1]);
				tmp_idle-=(t_c + t_min);
				//test_idle -= processorsLists[i].get(j).getProccesingTime();
			}
			//Calculate idle energy
			result += ScheduledTask.getEnergy(tmp_idle, processorPairs[i][processorPairs[i].length-1]);
			//total_idle+=test_idle;
			//if(debug_once){

			if(Double.isNaN(result)){
				logger_.severe("Energy is NaN!"+ ++count);
				//	}
			}
		}

		//System.out.println("Idle w. slack "+total_idle);
//		return total_idle;
		return result;
	}
//	protected void calculateSlackInSchedule(double makespan){
//		//System.out.println("Slack computing:");
//		for(int i = 0; i < tasksToSchedule.length; i++){
//			//Init variables
//			if(debug_once)System.out.println("Task "+i);
//			ScheduledTask tmp_st = tasksToSchedule[i];
//			//To be sure that slack is set
//			tmp_st.setSlack(-1);
//			double end_time = tmp_st.getEndTime();
//			double diff;
//			//Last task
//			if(processorsLists[tmp_st.getUsedProcessor()].size()==tmp_st.getPositionInProcessor()+1){
//				if(debug_once)System.out.println("Last task in proc "+ tmp_st.getUsedProcessor());
//				diff = makespan - end_time;
//			}
//			//Not last task
//			else {
//				ScheduledTask next_st = processorsLists[tmp_st.getUsedProcessor()].get(tmp_st.getPositionInProcessor()+1);
//				diff = next_st.getStartTime()-end_time;
//			}
//			//Check base diff
//			if(diff > small_value){
//				tmp_st.setSlack(diff);
//			} else {
//				tmp_st.setSlack(0);
//				if(debug_once)System.out.println("No slack, processor occupied");
//				tmp_st.setSlack(tmp_st.getSlack()+tmp_st.getProccesingTime());
//				continue;
//			}
//			Task tmp_t = tmp_st.getTask();
//
//			//Check succesors
//			for(int j=0;j<tmp_t.getNumOutEdges();j++){
//				diff = tmp_t.getOutEdge(j).getEndTask().getTmpScheduledTask().getStartTime()-tmp_t.getOutEdge(j).getCommTime()-end_time;
//				if(diff < tmp_st.getSlack()){
//					tmp_st.setSlack(diff);
//					if(diff < small_value){
//						if(debug_once)System.out.println("No slack, precedence constraint");
//						tmp_st.setSlack(0);
//						break;
//					}
//				}
//			}
//			if(debug_once)System.out.println("Slack: "+tmp_st.getSlack());
//			tmp_st.setSlack(tmp_st.getSlack()+tmp_st.getProccesingTime());
//
//		}
//
//	}
//
//}		
protected void calculateSlackInSchedule(double makespan){
	//System.out.println("Slack computing:");
	//	for(int i = 0; i < tasksToSchedule.length; i++){
	for(int i = 0; i < processorsLists.length; i++){
		for(int k = 0; k < processorsLists[i].size(); k++){
			//Init variables
			//if(debug_once)System.out.println("Task "+i);
			ScheduledTask tmp_st = processorsLists[i].get(k);//tasksToSchedule[i];
			//To be sure that slack is set
			tmp_st.setSlack(-1);
			double end_time = tmp_st.getEndTime();
			double diff;
			//Last task
			if(k==processorsLists[i].size()-1){
				if(debug_once)System.out.println("Last task in proc "+ tmp_st.getUsedProcessor());
				diff = makespan - end_time;
			}
			//Not last task
			else {
				ScheduledTask next_st = processorsLists[i].get(k+1);
				diff = next_st.getStartTime()-end_time;
			}
			//Check base diff
			if(diff > small_value){
				tmp_st.setSlack(diff);
			} else {
				tmp_st.setSlack(0);
				if(debug_once)System.out.println("No slack, processor occupied");
				tmp_st.setSlack(tmp_st.getSlack()+tmp_st.getProccesingTime());
				continue;
			}
			Task tmp_t = tmp_st.getTask();

			//Check succesors
			for(int j=0;j<tmp_t.getNumOutEdges();j++){
				diff = tmp_t.getOutEdge(j).getEndTask().getTmpScheduledTask().getStartTime()-tmp_t.getOutEdge(j).getCommTime()-end_time;
				if(diff < tmp_st.getSlack()){
					tmp_st.setSlack(diff);
					if(diff < small_value){
						if(debug_once)System.out.println("No slack, precedence constraint");
						tmp_st.setSlack(0);
						break;
					}
				}
			}
			if(debug_once)System.out.println("Slack: "+tmp_st.getSlack());
			tmp_st.setSlack(tmp_st.getSlack()+tmp_st.getProccesingTime());

		}
	}

}

}			





//Is it sink task?
//			if(tmp_t.getNumOutEdges()==0){
//				if(debug_once)System.out.println("Slack: "+tmp_st.getSlack());
//				if(debug_once)System.out.println("Last task in the schedule");
//				tmp_st.setSlack(tmp_st.getSlack()+tmp_st.getProccesingTime());
//				continue;
//			} 
//Task tmp_t = tmp_st.getTask();


//				if(diff > small_value){
//					tmp_st.setSlack(diff);
//				} else {
//					tmp_st.setSlack(0);
//					if(debug_once)System.out.println("Slack: "+tmp_st.getSlack());
//					if(debug_once)System.out.println("No slack, processor occupied");
//					tmp_st.setSlack(tmp_st.getSlack()+tmp_st.getProccesingTime());
//					continue;
//				}

//			if(test_evaluation){
//				System.out.format("T %d"+" P: %d"
//						+ " Vlvl: %.2f"
//						+" start: %.2f"
//						+" base: %.2f"
//						+" exec: %.2f"  
//						+ " end: %.2f\n",
//						tasksToSchedule[i].getTask().getIndex(),
//						tasksToSchedule[i].getUsedProcessor(),
//						tasksToSchedule[i].getVlsSettings().getrSpeed(),
//						tasksToSchedule[i].getStartTime(),
//						tasksToSchedule[i].getTask().getRunTime(tasksToSchedule[i].getUsedProcessor()),
//						tasksToSchedule[i].getProccesingTime(),
//						tasksToSchedule[i].getEndTime());
////				System.out.println("Task "+i+" on proc:"+tasksToSchedule[i].getUsedProcessor()
////						+ " V lvl: " + tasksToSchedule[i].getVlsSettings().getrSpeed()
////						+" start: "+ tasksToSchedule[i].getStartTime()
////						+" exec t: " + tasksToSchedule[i].getProccesingTime()
////						+ " end: "+ tasksToSchedule[i].getEndTime());
//			}
